<?php
/**
 * Plugin Name: WordFoundry Connect
 * Plugin URI: https://wordfoundry.app
 * Description: Connect your WordPress site to WordFoundry for AI-powered article publishing and two-way sync.
 * Version: 1.0.0
 * Author: WordFoundry
 * Author URI: https://wordfoundry.app
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: wordfoundry-connect
 * Domain Path: /languages
 * Requires at least: 6.0
 * Requires PHP: 7.4
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Plugin constants
define('WORDFOUNDRY_VERSION', '1.0.0');
define('WORDFOUNDRY_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('WORDFOUNDRY_PLUGIN_URL', plugin_dir_url(__FILE__));
define('WORDFOUNDRY_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * Main WordFoundry class
 */
class WordFoundry {
    /**
     * Single instance
     */
    private static $instance = null;

    /**
     * Get instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        $this->load_dependencies();
        $this->init_hooks();
    }

    /**
     * Load required files
     */
    private function load_dependencies() {
        require_once WORDFOUNDRY_PLUGIN_DIR . 'includes/class-wordfoundry-settings.php';
        require_once WORDFOUNDRY_PLUGIN_DIR . 'includes/class-wordfoundry-api.php';
        require_once WORDFOUNDRY_PLUGIN_DIR . 'includes/class-wordfoundry-sync.php';

        if (is_admin()) {
            require_once WORDFOUNDRY_PLUGIN_DIR . 'admin/class-wordfoundry-admin.php';
        }
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Load text domain
        add_action('plugins_loaded', array($this, 'load_textdomain'));

        // Register REST API endpoints
        add_action('rest_api_init', array('WordFoundry_API', 'register_routes'));

        // Two-way sync hooks
        add_action('save_post', array('WordFoundry_Sync', 'on_post_save'), 10, 3);
        add_action('before_delete_post', array('WordFoundry_Sync', 'on_post_delete'), 10, 1);
        add_action('wp_trash_post', array('WordFoundry_Sync', 'on_post_trash'), 10, 1);
        add_action('untrash_post', array('WordFoundry_Sync', 'on_post_untrash'), 10, 1);

        // Admin hooks
        if (is_admin()) {
            WordFoundry_Admin::init();
        }
    }

    /**
     * Load plugin text domain
     */
    public function load_textdomain() {
        load_plugin_textdomain(
            'wordfoundry-connect',
            false,
            dirname(WORDFOUNDRY_PLUGIN_BASENAME) . '/languages/'
        );
    }

    /**
     * Activation hook
     */
    public static function activate() {
        // Create options with defaults
        if (!get_option('wordfoundry_settings')) {
            add_option('wordfoundry_settings', array(
                'api_url' => 'https://wordfoundry.app',
                'post_type' => 'post',
                'taxonomy' => 'category',
                'default_status' => 'draft',
                'sync_enabled' => true,
            ));
        }

        // Flush rewrite rules
        flush_rewrite_rules();
    }

    /**
     * Deactivation hook
     */
    public static function deactivate() {
        flush_rewrite_rules();
    }
}

// Activation/deactivation hooks
register_activation_hook(__FILE__, array('WordFoundry', 'activate'));
register_deactivation_hook(__FILE__, array('WordFoundry', 'deactivate'));

// Initialize plugin
function wordfoundry_init() {
    return WordFoundry::get_instance();
}
add_action('plugins_loaded', 'wordfoundry_init');
