<?php
/**
 * WordFoundry Two-Way Sync Handler
 *
 * Handles synchronization between WordPress and WordFoundry:
 * - Sends webhook to WordFoundry when post is updated in WordPress
 * - Prevents sync loops using meta flags
 */

if (!defined('ABSPATH')) {
    exit;
}

class WordFoundry_Sync {
    /**
     * Hook into post save
     */
    public static function on_post_save($post_id, $post, $update) {
        // Skip if not an update (new post)
        if (!$update) {
            return;
        }

        // Skip autosaves
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }

        // Skip revisions
        if (wp_is_post_revision($post_id)) {
            return;
        }

        // Skip if sync is disabled
        $settings = WordFoundry_Settings::get_all();
        if (empty($settings['sync_enabled'])) {
            return;
        }

        // Check if this post type is tracked
        if ($post->post_type !== $settings['post_type']) {
            return;
        }

        // Skip if not configured
        if (!WordFoundry_Settings::is_configured()) {
            return;
        }

        // Check if this post was created/synced from WordFoundry
        $wordfoundry_article_id = get_post_meta($post_id, '_wordfoundry_article_id', true);
        if (empty($wordfoundry_article_id)) {
            return; // Not a WordFoundry article
        }

        // Prevent sync loops - check if this update came from WordFoundry
        $last_synced = get_post_meta($post_id, '_wordfoundry_synced', true);
        if (!empty($last_synced)) {
            // Clear the flag and skip this sync
            delete_post_meta($post_id, '_wordfoundry_synced');
            return;
        }

        // Send update to WordFoundry
        self::send_to_wordfoundry($post_id, $post);
    }

    /**
     * Send post update to WordFoundry via webhook
     */
    private static function send_to_wordfoundry($post_id, $post) {
        $settings = WordFoundry_Settings::get_all();
        $webhook_url = WordFoundry_Settings::get_webhook_url();

        if (empty($webhook_url)) {
            return;
        }

        // Prepare payload
        $payload = array(
            'event' => 'post.updated',
            'post_id' => $post_id,
            'title' => $post->post_title,
            'content' => $post->post_content,
            'status' => $post->post_status,
            'modified_at' => get_post_modified_time('c', true, $post_id),
        );

        $body = wp_json_encode($payload);

        // Create signature for verification
        $signature = '';
        if (!empty($settings['webhook_secret'])) {
            $signature = 'sha256=' . hash_hmac('sha256', $body, $settings['webhook_secret']);
        }

        // Send webhook
        $response = wp_remote_post($webhook_url, array(
            'method' => 'POST',
            'timeout' => 30,
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-WF-Signature' => $signature,
            ),
            'body' => $body,
        ));

        // Log errors for debugging
        if (is_wp_error($response)) {
            error_log('WordFoundry Sync Error: ' . $response->get_error_message());
        } else {
            $status_code = wp_remote_retrieve_response_code($response);
            if ($status_code >= 400) {
                error_log('WordFoundry Sync Error: HTTP ' . $status_code);
            }
        }
    }

    /**
     * Hook into post delete (before deletion)
     */
    public static function on_post_delete($post_id) {
        $post = get_post($post_id);
        if (!$post) {
            return;
        }

        // Skip if sync is disabled
        $settings = WordFoundry_Settings::get_all();
        if (empty($settings['sync_enabled'])) {
            return;
        }

        // Check if this post type is tracked
        if ($post->post_type !== $settings['post_type']) {
            return;
        }

        // Skip if not configured
        if (!WordFoundry_Settings::is_configured()) {
            return;
        }

        // Check if this post was created/synced from WordFoundry
        $wordfoundry_article_id = get_post_meta($post_id, '_wordfoundry_article_id', true);
        if (empty($wordfoundry_article_id)) {
            return; // Not a WordFoundry article
        }

        // Send delete notification to WordFoundry
        self::send_delete_to_wordfoundry($post_id, $wordfoundry_article_id);
    }

    /**
     * Hook into post trash
     */
    public static function on_post_trash($post_id) {
        $post = get_post($post_id);
        if (!$post) {
            return;
        }

        // Skip if sync is disabled
        $settings = WordFoundry_Settings::get_all();
        if (empty($settings['sync_enabled'])) {
            return;
        }

        // Check if this post type is tracked
        if ($post->post_type !== $settings['post_type']) {
            return;
        }

        // Skip if not configured
        if (!WordFoundry_Settings::is_configured()) {
            return;
        }

        // Check if this post was created/synced from WordFoundry
        $wordfoundry_article_id = get_post_meta($post_id, '_wordfoundry_article_id', true);
        if (empty($wordfoundry_article_id)) {
            return; // Not a WordFoundry article
        }

        // Send trash notification to WordFoundry
        self::send_event_to_wordfoundry('post.trashed', $post_id, $wordfoundry_article_id);
    }

    /**
     * Hook into post untrash (restore from trash)
     */
    public static function on_post_untrash($post_id) {
        $post = get_post($post_id);
        if (!$post) {
            return;
        }

        // Skip if sync is disabled
        $settings = WordFoundry_Settings::get_all();
        if (empty($settings['sync_enabled'])) {
            return;
        }

        // Check if this post type is tracked
        if ($post->post_type !== $settings['post_type']) {
            return;
        }

        // Skip if not configured
        if (!WordFoundry_Settings::is_configured()) {
            return;
        }

        // Check if this post was created/synced from WordFoundry
        $wordfoundry_article_id = get_post_meta($post_id, '_wordfoundry_article_id', true);
        if (empty($wordfoundry_article_id)) {
            return; // Not a WordFoundry article
        }

        // Get the status the post will be restored to (stored in meta before trashing)
        $restored_status = get_post_meta($post_id, '_wp_trash_meta_status', true);
        if (empty($restored_status)) {
            $restored_status = 'draft';
        }

        // Send restore notification to WordFoundry
        self::send_event_to_wordfoundry('post.restored', $post_id, $wordfoundry_article_id, $restored_status);
    }

    /**
     * Send event notification to WordFoundry via webhook
     */
    private static function send_event_to_wordfoundry($event, $post_id, $article_id, $status = null) {
        $settings = WordFoundry_Settings::get_all();
        $webhook_url = WordFoundry_Settings::get_webhook_url();

        if (empty($webhook_url)) {
            return;
        }

        // Prepare payload
        $payload = array(
            'event' => $event,
            'post_id' => $post_id,
            'article_id' => intval($article_id),
            'timestamp' => current_time('c'),
        );

        if ($status !== null) {
            $payload['status'] = $status;
        }

        $body = wp_json_encode($payload);

        // Create signature for verification
        $signature = '';
        if (!empty($settings['webhook_secret'])) {
            $signature = 'sha256=' . hash_hmac('sha256', $body, $settings['webhook_secret']);
        }

        // Send webhook
        $response = wp_remote_post($webhook_url, array(
            'method' => 'POST',
            'timeout' => 30,
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-WF-Signature' => $signature,
            ),
            'body' => $body,
        ));

        // Log errors for debugging
        if (is_wp_error($response)) {
            error_log('WordFoundry Sync ' . $event . ' Error: ' . $response->get_error_message());
        } else {
            $status_code = wp_remote_retrieve_response_code($response);
            if ($status_code >= 400) {
                error_log('WordFoundry Sync ' . $event . ' Error: HTTP ' . $status_code);
            }
        }
    }

    /**
     * Send post delete notification to WordFoundry via webhook
     */
    private static function send_delete_to_wordfoundry($post_id, $article_id) {
        $settings = WordFoundry_Settings::get_all();
        $webhook_url = WordFoundry_Settings::get_webhook_url();

        if (empty($webhook_url)) {
            return;
        }

        // Prepare payload
        $payload = array(
            'event' => 'post.deleted',
            'post_id' => $post_id,
            'article_id' => intval($article_id),
            'deleted_at' => current_time('c'),
        );

        $body = wp_json_encode($payload);

        // Create signature for verification
        $signature = '';
        if (!empty($settings['webhook_secret'])) {
            $signature = 'sha256=' . hash_hmac('sha256', $body, $settings['webhook_secret']);
        }

        // Send webhook
        $response = wp_remote_post($webhook_url, array(
            'method' => 'POST',
            'timeout' => 30,
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-WF-Signature' => $signature,
            ),
            'body' => $body,
        ));

        // Log errors for debugging
        if (is_wp_error($response)) {
            error_log('WordFoundry Sync Delete Error: ' . $response->get_error_message());
        } else {
            $status_code = wp_remote_retrieve_response_code($response);
            if ($status_code >= 400) {
                error_log('WordFoundry Sync Delete Error: HTTP ' . $status_code);
            }
        }
    }

    /**
     * Manual sync trigger (for admin UI)
     */
    public static function manual_sync($post_id) {
        $post = get_post($post_id);
        if (!$post) {
            return new WP_Error('not_found', 'Post not found');
        }

        self::send_to_wordfoundry($post_id, $post);
        return true;
    }
}
