<?php
/**
 * WordFoundry REST API Endpoints
 *
 * Provides endpoints for WordFoundry to interact with WordPress:
 * - POST /wordfoundry/v1/publish - Publish an article
 * - PUT /wordfoundry/v1/posts/{id} - Update an article
 * - DELETE /wordfoundry/v1/posts/{id} - Trash an article
 * - POST /wordfoundry/v1/posts/{id}/restore - Restore from trash
 * - GET /wordfoundry/v1/posts - List posts for import
 * - GET /wordfoundry/v1/posts/{id}/export - Get post content for import
 * - GET /wordfoundry/v1/settings - Get site settings
 * - POST /wordfoundry/v1/verify - Verify connection
 * - GET /wordfoundry/v1/categories - Get categories list
 * - GET /wordfoundry/v1/post-types - Get available post types
 * - GET /wordfoundry/v1/taxonomies/{post_type} - Get taxonomies for post type
 * - GET /wordfoundry/v1/terms/{taxonomy} - Get terms for taxonomy
 */

if (!defined('ABSPATH')) {
    exit;
}

class WordFoundry_API {
    /**
     * Namespace
     */
    const NAMESPACE = 'wordfoundry/v1';

    /**
     * Register REST routes
     */
    public static function register_routes() {
        // Verify connection
        register_rest_route(self::NAMESPACE, '/verify', array(
            'methods' => 'POST',
            'callback' => array(__CLASS__, 'verify_connection'),
            'permission_callback' => array(__CLASS__, 'check_permission'),
        ));

        // Get site settings
        register_rest_route(self::NAMESPACE, '/settings', array(
            'methods' => 'GET',
            'callback' => array(__CLASS__, 'get_settings'),
            'permission_callback' => array(__CLASS__, 'check_permission'),
        ));

        // Get categories
        register_rest_route(self::NAMESPACE, '/categories', array(
            'methods' => 'GET',
            'callback' => array(__CLASS__, 'get_categories'),
            'permission_callback' => array(__CLASS__, 'check_permission'),
        ));

        // Publish article
        register_rest_route(self::NAMESPACE, '/publish', array(
            'methods' => 'POST',
            'callback' => array(__CLASS__, 'publish_article'),
            'permission_callback' => array(__CLASS__, 'check_permission'),
        ));

        // Update article
        register_rest_route(self::NAMESPACE, '/posts/(?P<id>\d+)', array(
            'methods' => 'PUT',
            'callback' => array(__CLASS__, 'update_article'),
            'permission_callback' => array(__CLASS__, 'check_permission'),
            'args' => array(
                'id' => array(
                    'validate_callback' => function($param) {
                        return is_numeric($param);
                    }
                ),
            ),
        ));

        // Delete/trash article
        register_rest_route(self::NAMESPACE, '/posts/(?P<id>\d+)', array(
            'methods' => 'DELETE',
            'callback' => array(__CLASS__, 'delete_article'),
            'permission_callback' => array(__CLASS__, 'check_permission'),
            'args' => array(
                'id' => array(
                    'validate_callback' => function($param) {
                        return is_numeric($param);
                    }
                ),
            ),
        ));

        // Restore article from trash
        register_rest_route(self::NAMESPACE, '/posts/(?P<id>\d+)/restore', array(
            'methods' => 'POST',
            'callback' => array(__CLASS__, 'restore_article'),
            'permission_callback' => array(__CLASS__, 'check_permission'),
            'args' => array(
                'id' => array(
                    'validate_callback' => function($param) {
                        return is_numeric($param);
                    }
                ),
            ),
        ));

        // List posts for import
        register_rest_route(self::NAMESPACE, '/posts', array(
            'methods' => 'GET',
            'callback' => array(__CLASS__, 'list_posts'),
            'permission_callback' => array(__CLASS__, 'check_permission'),
            'args' => array(
                'status' => array(
                    'default' => 'any',
                    'sanitize_callback' => 'sanitize_text_field',
                ),
                'per_page' => array(
                    'default' => 20,
                    'sanitize_callback' => 'absint',
                ),
                'page' => array(
                    'default' => 1,
                    'sanitize_callback' => 'absint',
                ),
                'search' => array(
                    'default' => '',
                    'sanitize_callback' => 'sanitize_text_field',
                ),
                'exclude_imported' => array(
                    'default' => 'true',
                    'sanitize_callback' => 'sanitize_text_field',
                ),
            ),
        ));

        // Get single post for import
        register_rest_route(self::NAMESPACE, '/posts/(?P<id>\d+)/export', array(
            'methods' => 'GET',
            'callback' => array(__CLASS__, 'get_post_for_export'),
            'permission_callback' => array(__CLASS__, 'check_permission'),
            'args' => array(
                'id' => array(
                    'validate_callback' => function($param) {
                        return is_numeric($param);
                    }
                ),
            ),
        ));

        // Get available post types
        register_rest_route(self::NAMESPACE, '/post-types', array(
            'methods' => 'GET',
            'callback' => array(__CLASS__, 'get_post_types'),
            'permission_callback' => array(__CLASS__, 'check_permission'),
        ));

        // Get taxonomies for post type
        register_rest_route(self::NAMESPACE, '/taxonomies/(?P<post_type>[a-zA-Z0-9_-]+)', array(
            'methods' => 'GET',
            'callback' => array(__CLASS__, 'get_taxonomies'),
            'permission_callback' => array(__CLASS__, 'check_permission'),
            'args' => array(
                'post_type' => array(
                    'sanitize_callback' => 'sanitize_key',
                ),
            ),
        ));

        // Get terms for taxonomy
        register_rest_route(self::NAMESPACE, '/terms/(?P<taxonomy>[a-zA-Z0-9_-]+)', array(
            'methods' => 'GET',
            'callback' => array(__CLASS__, 'get_taxonomy_terms'),
            'permission_callback' => array(__CLASS__, 'check_permission'),
            'args' => array(
                'taxonomy' => array(
                    'sanitize_callback' => 'sanitize_key',
                ),
                'per_page' => array(
                    'default' => 100,
                    'sanitize_callback' => 'absint',
                ),
                'search' => array(
                    'default' => '',
                    'sanitize_callback' => 'sanitize_text_field',
                ),
            ),
        ));
    }

    /**
     * Check permission for API access
     * Uses WordPress Application Passwords (Basic Auth)
     */
    public static function check_permission($request) {
        // WordPress handles Basic Auth automatically with Application Passwords
        // If user is authenticated, they have permission
        return is_user_logged_in() && current_user_can('edit_posts');
    }

    /**
     * Verify connection endpoint
     */
    public static function verify_connection($request) {
        return new WP_REST_Response(array(
            'success' => true,
            'site_name' => get_bloginfo('name'),
            'message' => 'Connection verified successfully',
            'wordpress_version' => get_bloginfo('version'),
            'plugin_version' => WORDFOUNDRY_VERSION,
        ), 200);
    }

    /**
     * Get site settings
     */
    public static function get_settings($request) {
        $settings = WordFoundry_Settings::get_all();

        // Get default categories (for backwards compatibility)
        $categories = get_terms(array(
            'taxonomy' => 'category',
            'hide_empty' => false,
        ));

        $categories_data = array();
        if (!is_wp_error($categories)) {
            foreach ($categories as $cat) {
                $categories_data[] = array(
                    'id' => $cat->term_id,
                    'name' => $cat->name,
                    'slug' => $cat->slug,
                );
            }
        }

        // Get available post types
        $post_types = get_post_types(array('public' => true), 'objects');
        $post_types_data = array();
        foreach ($post_types as $pt) {
            if ($pt->name === 'attachment') continue;
            $post_types_data[] = array(
                'name' => $pt->name,
                'label' => $pt->labels->singular_name,
            );
        }

        return new WP_REST_Response(array(
            'site_name' => get_bloginfo('name'),
            'cms' => 'wordpress',
            'cms_version' => get_bloginfo('version'),
            'plugin_version' => WORDFOUNDRY_VERSION,
            'post_types' => $post_types_data,
            'categories' => $categories_data,
            'sync_enabled' => $settings['sync_enabled'],
        ), 200);
    }

    /**
     * Get categories list
     */
    public static function get_categories($request) {
        $settings = WordFoundry_Settings::get_all();
        $taxonomy = $request->get_param('taxonomy') ?: 'category';

        $terms = get_terms(array(
            'taxonomy' => $taxonomy,
            'hide_empty' => false,
            'number' => 100,
        ));

        if (is_wp_error($terms)) {
            return new WP_REST_Response(array(
                'success' => false,
                'message' => $terms->get_error_message(),
            ), 400);
        }

        $categories = array();
        foreach ($terms as $term) {
            $categories[] = array(
                'id' => $term->term_id,
                'name' => $term->name,
                'slug' => $term->slug,
                'count' => $term->count,
            );
        }

        return new WP_REST_Response(array(
            'categories' => $categories,
        ), 200);
    }

    /**
     * Get available post types
     */
    public static function get_post_types($request) {
        $post_types = get_post_types(array('public' => true), 'objects');

        $result = array();
        foreach ($post_types as $post_type) {
            // Skip attachments
            if ($post_type->name === 'attachment') {
                continue;
            }

            $result[] = array(
                'name' => $post_type->name,
                'label' => $post_type->labels->singular_name,
                'label_plural' => $post_type->labels->name,
                'description' => $post_type->description,
                'hierarchical' => $post_type->hierarchical,
                'has_archive' => $post_type->has_archive,
                'supports' => get_all_post_type_supports($post_type->name),
            );
        }

        return new WP_REST_Response(array(
            'success' => true,
            'post_types' => $result,
        ), 200);
    }

    /**
     * Get taxonomies for a post type
     */
    public static function get_taxonomies($request) {
        $post_type = $request['post_type'];

        // Check if post type exists
        if (!post_type_exists($post_type)) {
            return new WP_REST_Response(array(
                'success' => false,
                'error' => array(
                    'code' => 'INVALID_POST_TYPE',
                    'message' => 'Post type does not exist',
                ),
            ), 400);
        }

        $taxonomies = get_object_taxonomies($post_type, 'objects');

        $result = array();
        foreach ($taxonomies as $taxonomy) {
            // Skip non-public taxonomies
            if (!$taxonomy->public) {
                continue;
            }

            $result[] = array(
                'name' => $taxonomy->name,
                'label' => $taxonomy->labels->singular_name,
                'label_plural' => $taxonomy->labels->name,
                'hierarchical' => $taxonomy->hierarchical,
                'show_ui' => $taxonomy->show_ui,
            );
        }

        return new WP_REST_Response(array(
            'success' => true,
            'post_type' => $post_type,
            'taxonomies' => $result,
        ), 200);
    }

    /**
     * Get terms for a taxonomy
     */
    public static function get_taxonomy_terms($request) {
        $taxonomy = $request['taxonomy'];
        $per_page = min(absint($request->get_param('per_page')), 500);
        $search = $request->get_param('search');

        // Check if taxonomy exists
        if (!taxonomy_exists($taxonomy)) {
            return new WP_REST_Response(array(
                'success' => false,
                'error' => array(
                    'code' => 'INVALID_TAXONOMY',
                    'message' => 'Taxonomy does not exist',
                ),
            ), 400);
        }

        $args = array(
            'taxonomy' => $taxonomy,
            'hide_empty' => false,
            'number' => $per_page,
        );

        if (!empty($search)) {
            $args['search'] = $search;
        }

        $terms = get_terms($args);

        if (is_wp_error($terms)) {
            return new WP_REST_Response(array(
                'success' => false,
                'error' => array(
                    'code' => 'TERM_ERROR',
                    'message' => $terms->get_error_message(),
                ),
            ), 400);
        }

        $result = array();
        foreach ($terms as $term) {
            $result[] = array(
                'id' => $term->term_id,
                'name' => $term->name,
                'slug' => $term->slug,
                'count' => $term->count,
                'parent' => $term->parent,
            );
        }

        // Get taxonomy object for label
        $taxonomy_obj = get_taxonomy($taxonomy);

        return new WP_REST_Response(array(
            'success' => true,
            'taxonomy' => $taxonomy,
            'taxonomy_label' => $taxonomy_obj->labels->singular_name,
            'terms' => $result,
            'total' => count($result),
        ), 200);
    }

    /**
     * Publish article from WordFoundry
     */
    public static function publish_article($request) {
        $params = $request->get_json_params();

        // Required fields
        if (empty($params['title']) || empty($params['content'])) {
            return new WP_REST_Response(array(
                'success' => false,
                'error' => array(
                    'code' => 'MISSING_FIELDS',
                    'message' => 'Title and content are required',
                ),
            ), 400);
        }

        // Get post_type from request or default to 'post'
        $post_type = isset($params['post_type']) ? sanitize_key($params['post_type']) : 'post';

        // Validate post type exists
        if (!post_type_exists($post_type)) {
            return new WP_REST_Response(array(
                'success' => false,
                'error' => array(
                    'code' => 'INVALID_POST_TYPE',
                    'message' => 'Post type does not exist',
                ),
            ), 400);
        }

        // Process content - upload images and replace URLs
        $processed_content = self::process_content_images($params['content']);

        // Prepare post data
        $post_data = array(
            'post_title' => sanitize_text_field($params['title']),
            'post_content' => wp_kses_post($processed_content),
            'post_status' => isset($params['status']) ? sanitize_text_field($params['status']) : 'draft',
            'post_type' => $post_type,
        );

        // Set post author if provided
        if (!empty($params['author']) && is_numeric($params['author'])) {
            $author_id = intval($params['author']);
            // Verify the user exists and can publish posts
            $author = get_user_by('id', $author_id);
            if ($author && user_can($author, 'edit_posts')) {
                $post_data['post_author'] = $author_id;
            }
        }

        // Meta description (for Yoast SEO or similar)
        $meta_description = isset($params['meta_description']) ? sanitize_text_field($params['meta_description']) : '';

        // Create the post
        $post_id = wp_insert_post($post_data, true);

        if (is_wp_error($post_id)) {
            return new WP_REST_Response(array(
                'success' => false,
                'error' => array(
                    'code' => 'CREATE_FAILED',
                    'message' => $post_id->get_error_message(),
                ),
            ), 500);
        }

        // Store WordFoundry article ID
        if (!empty($params['article_id'])) {
            update_post_meta($post_id, '_wordfoundry_article_id', intval($params['article_id']));
        }

        // Set terms (categories/taxonomies)
        // Accept terms as array with taxonomy info: { taxonomy: 'category', terms: [1, 2, 3] }
        if (!empty($params['terms']) && is_array($params['terms'])) {
            foreach ($params['terms'] as $term_data) {
                if (!empty($term_data['taxonomy']) && !empty($term_data['term_ids'])) {
                    $taxonomy = sanitize_key($term_data['taxonomy']);
                    if (taxonomy_exists($taxonomy)) {
                        $term_ids = array_map('intval', (array) $term_data['term_ids']);
                        wp_set_post_terms($post_id, $term_ids, $taxonomy);
                    }
                }
            }
        }
        // Backwards compatibility: also accept categories as simple array
        elseif (!empty($params['categories'])) {
            $taxonomy = isset($params['taxonomy']) ? sanitize_key($params['taxonomy']) : 'category';
            $term_ids = array();
            foreach ($params['categories'] as $cat) {
                if (is_numeric($cat)) {
                    $term_ids[] = intval($cat);
                } else {
                    // Find by slug
                    $term = get_term_by('slug', $cat, $taxonomy);
                    if ($term) {
                        $term_ids[] = $term->term_id;
                    }
                }
            }
            if (!empty($term_ids)) {
                wp_set_post_terms($post_id, $term_ids, $taxonomy);
            }
        }

        // Set tags if provided (creates new tags if they don't exist)
        if (!empty($params['tags']) && is_array($params['tags'])) {
            wp_set_post_tags($post_id, $params['tags']);
        }

        // Set meta description (Yoast SEO)
        if (!empty($meta_description)) {
            update_post_meta($post_id, '_yoast_wpseo_metadesc', $meta_description);
        }

        // Handle featured image URL
        if (!empty($params['featured_image_url'])) {
            $image_id = self::upload_media_from_url($params['featured_image_url'], $post_id);
            if ($image_id) {
                set_post_thumbnail($post_id, $image_id);
            }
        }

        // Mark as synced from WordFoundry (to avoid sync loop)
        update_post_meta($post_id, '_wordfoundry_synced', current_time('mysql'));

        return new WP_REST_Response(array(
            'success' => true,
            'post_id' => $post_id,
            'permalink' => get_permalink($post_id),
            'status' => get_post_status($post_id),
        ), 201);
    }

    /**
     * Update existing article
     */
    public static function update_article($request) {
        $post_id = intval($request['id']);
        $params = $request->get_json_params();

        // Check if post exists
        $post = get_post($post_id);
        if (!$post) {
            return new WP_REST_Response(array(
                'success' => false,
                'error' => array(
                    'code' => 'NOT_FOUND',
                    'message' => 'Post not found',
                ),
            ), 404);
        }

        // Prepare update data
        $update_data = array('ID' => $post_id);

        if (isset($params['title'])) {
            $update_data['post_title'] = sanitize_text_field($params['title']);
        }
        if (isset($params['content'])) {
            // Process content - upload images and videos to WordPress Media Library
            $processed_content = self::process_content_images($params['content']);
            $update_data['post_content'] = wp_kses_post($processed_content);
        }
        if (isset($params['status'])) {
            $update_data['post_status'] = sanitize_text_field($params['status']);
        }

        // Mark as synced from WordFoundry (to avoid sync loop)
        update_post_meta($post_id, '_wordfoundry_synced', current_time('mysql'));

        // Update post
        $result = wp_update_post($update_data, true);

        if (is_wp_error($result)) {
            return new WP_REST_Response(array(
                'success' => false,
                'error' => array(
                    'code' => 'UPDATE_FAILED',
                    'message' => $result->get_error_message(),
                ),
            ), 500);
        }

        // Update meta description if provided
        if (isset($params['meta_description'])) {
            update_post_meta($post_id, '_yoast_wpseo_metadesc', sanitize_text_field($params['meta_description']));
        }

        return new WP_REST_Response(array(
            'success' => true,
            'post_id' => $post_id,
            'permalink' => get_permalink($post_id),
            'updated_at' => get_post_modified_time('c', true, $post_id),
        ), 200);
    }

    /**
     * Delete/trash article
     */
    public static function delete_article($request) {
        $post_id = intval($request['id']);

        // Check if post exists
        $post = get_post($post_id);
        if (!$post) {
            return new WP_REST_Response(array(
                'success' => false,
                'error' => array(
                    'code' => 'NOT_FOUND',
                    'message' => 'Post not found',
                ),
            ), 404);
        }

        // Move to trash (not permanent delete)
        $result = wp_trash_post($post_id);

        if (!$result) {
            return new WP_REST_Response(array(
                'success' => false,
                'error' => array(
                    'code' => 'DELETE_FAILED',
                    'message' => 'Failed to move post to trash',
                ),
            ), 500);
        }

        return new WP_REST_Response(array(
            'success' => true,
            'post_id' => $post_id,
            'status' => 'trashed',
        ), 200);
    }

    /**
     * Restore article from trash
     */
    public static function restore_article($request) {
        $post_id = intval($request['id']);

        // Check if post exists (including in trash)
        $post = get_post($post_id);
        if (!$post) {
            return new WP_REST_Response(array(
                'success' => false,
                'error' => array(
                    'code' => 'NOT_FOUND',
                    'message' => 'Post not found',
                ),
            ), 404);
        }

        // Check if post is in trash
        if ($post->post_status !== 'trash') {
            return new WP_REST_Response(array(
                'success' => false,
                'error' => array(
                    'code' => 'NOT_TRASHED',
                    'message' => 'Post is not in trash',
                ),
            ), 400);
        }

        // Restore from trash
        $result = wp_untrash_post($post_id);

        if (!$result) {
            return new WP_REST_Response(array(
                'success' => false,
                'error' => array(
                    'code' => 'RESTORE_FAILED',
                    'message' => 'Failed to restore post from trash',
                ),
            ), 500);
        }

        // Mark as synced from WordFoundry (to avoid sync loop)
        update_post_meta($post_id, '_wordfoundry_synced', current_time('mysql'));

        // Get the restored post to return its new status
        $restored_post = get_post($post_id);

        return new WP_REST_Response(array(
            'success' => true,
            'post_id' => $post_id,
            'status' => $restored_post->post_status,
            'permalink' => get_permalink($post_id),
        ), 200);
    }

    /**
     * List posts for import to WordFoundry
     */
    public static function list_posts($request) {
        $settings = WordFoundry_Settings::get_all();

        $status = $request->get_param('status');
        $per_page = min(absint($request->get_param('per_page')), 100);
        $page = max(1, absint($request->get_param('page')));
        $search = $request->get_param('search');
        $exclude_imported = $request->get_param('exclude_imported') === 'true';

        // Build query args
        $args = array(
            'post_type' => $settings['post_type'],
            'post_status' => $status === 'any' ? array('publish', 'draft', 'pending', 'private') : $status,
            'posts_per_page' => $per_page,
            'paged' => $page,
            'orderby' => 'modified',
            'order' => 'DESC',
        );

        // Add search
        if (!empty($search)) {
            $args['s'] = $search;
        }

        // Exclude already imported posts
        if ($exclude_imported) {
            $args['meta_query'] = array(
                'relation' => 'OR',
                array(
                    'key' => '_wordfoundry_article_id',
                    'compare' => 'NOT EXISTS',
                ),
                array(
                    'key' => '_wordfoundry_article_id',
                    'value' => '',
                    'compare' => '=',
                ),
            );
        }

        $query = new WP_Query($args);
        $posts = array();

        foreach ($query->posts as $post) {
            // Get author info
            $author = get_userdata($post->post_author);
            $author_name = $author ? $author->display_name : 'Unknown';

            // Get featured image
            $featured_image_url = '';
            $thumbnail_id = get_post_thumbnail_id($post->ID);
            if ($thumbnail_id) {
                $featured_image_url = wp_get_attachment_url($thumbnail_id);
            }

            // Get excerpt
            $excerpt = $post->post_excerpt;
            if (empty($excerpt)) {
                $excerpt = wp_trim_words(strip_tags($post->post_content), 30, '...');
            }

            $posts[] = array(
                'id' => $post->ID,
                'title' => $post->post_title,
                'excerpt' => $excerpt,
                'status' => $post->post_status,
                'date' => get_post_time('c', true, $post->ID),
                'modified' => get_post_modified_time('c', true, $post->ID),
                'author' => $author_name,
                'author_id' => $post->post_author,
                'featured_image_url' => $featured_image_url,
                'permalink' => get_permalink($post->ID),
                'word_count' => str_word_count(strip_tags($post->post_content)),
            );
        }

        return new WP_REST_Response(array(
            'success' => true,
            'posts' => $posts,
            'total' => $query->found_posts,
            'pages' => $query->max_num_pages,
            'current_page' => $page,
        ), 200);
    }

    /**
     * Get single post content for export/import to WordFoundry
     */
    public static function get_post_for_export($request) {
        $post_id = intval($request['id']);

        $post = get_post($post_id);
        if (!$post) {
            return new WP_REST_Response(array(
                'success' => false,
                'error' => array(
                    'code' => 'NOT_FOUND',
                    'message' => 'Post not found',
                ),
            ), 404);
        }

        // Get author info
        $author = get_userdata($post->post_author);
        $author_name = $author ? $author->display_name : 'Unknown';

        // Get featured image
        $featured_image_url = '';
        $thumbnail_id = get_post_thumbnail_id($post->ID);
        if ($thumbnail_id) {
            $featured_image_url = wp_get_attachment_url($thumbnail_id);
        }

        // Get categories
        $settings = WordFoundry_Settings::get_all();
        $terms = wp_get_post_terms($post->ID, $settings['taxonomy']);
        $categories = array();
        if (!is_wp_error($terms)) {
            foreach ($terms as $term) {
                $categories[] = array(
                    'id' => $term->term_id,
                    'name' => $term->name,
                    'slug' => $term->slug,
                );
            }
        }

        // Get tags
        $tags = wp_get_post_tags($post->ID);
        $tag_list = array();
        if (!is_wp_error($tags)) {
            foreach ($tags as $tag) {
                $tag_list[] = $tag->name;
            }
        }

        // Get meta description (Yoast SEO)
        $meta_description = get_post_meta($post->ID, '_yoast_wpseo_metadesc', true);

        return new WP_REST_Response(array(
            'success' => true,
            'post' => array(
                'id' => $post->ID,
                'title' => $post->post_title,
                'content' => $post->post_content,
                'excerpt' => $post->post_excerpt,
                'status' => $post->post_status,
                'date' => get_post_time('c', true, $post->ID),
                'modified' => get_post_modified_time('c', true, $post->ID),
                'author' => $author_name,
                'author_id' => $post->post_author,
                'featured_image_url' => $featured_image_url,
                'permalink' => get_permalink($post->ID),
                'word_count' => str_word_count(strip_tags($post->post_content)),
                'categories' => $categories,
                'tags' => $tag_list,
                'meta_description' => $meta_description,
            ),
        ), 200);
    }

    /**
     * Process content media (images and videos) - upload to WordPress Media Library and replace URLs
     */
    private static function process_content_images($content) {
        // First convert markdown images to HTML
        // Pattern: ![alt text](url)
        $content = preg_replace_callback(
            '/!\[([^\]]*)\]\(([^)]+)\)/',
            function($matches) {
                $alt = esc_attr($matches[1]);
                $url = esc_url($matches[2]);
                return '<img src="' . $url . '" alt="' . $alt . '" />';
            },
            $content
        );

        // Process all img tags and upload images
        // Pattern matches <img ... src="url" ... />
        $content = preg_replace_callback(
            '/<img([^>]*)src=["\']([^"\']+)["\']([^>]*)>/i',
            function($matches) {
                $before_src = $matches[1];
                $image_url = $matches[2];
                $after_src = $matches[3];

                // Skip if already a WordPress URL or data URL
                $site_url = get_site_url();
                if (strpos($image_url, $site_url) === 0 || strpos($image_url, 'data:') === 0) {
                    return $matches[0];
                }

                // Make URL absolute if relative
                if (strpos($image_url, 'http') !== 0) {
                    // Assume it's from WordFoundry server
                    $wordfoundry_url = WordFoundry_Settings::get('api_url', 'https://wordfoundry.app');
                    $image_url = rtrim($wordfoundry_url, '/') . '/' . ltrim($image_url, '/');
                }

                // Upload image to WordPress Media Library
                $attachment_id = self::upload_media_from_url($image_url);

                if ($attachment_id) {
                    // Get the new WordPress URL
                    $new_url = wp_get_attachment_url($attachment_id);
                    if ($new_url) {
                        return '<img' . $before_src . 'src="' . esc_url($new_url) . '"' . $after_src . '>';
                    }
                }

                // If upload failed, keep original
                return $matches[0];
            },
            $content
        );

        // Process all video tags and upload videos
        // Pattern matches <video ... src="url" ... />
        $content = preg_replace_callback(
            '/<video([^>]*)src=["\']([^"\']+)["\']([^>]*)>/i',
            function($matches) {
                $before_src = $matches[1];
                $video_url = $matches[2];
                $after_src = $matches[3];

                // Skip if already a WordPress URL
                $site_url = get_site_url();
                if (strpos($video_url, $site_url) === 0) {
                    return $matches[0];
                }

                // Make URL absolute if relative
                if (strpos($video_url, 'http') !== 0) {
                    $wordfoundry_url = WordFoundry_Settings::get('api_url', 'https://wordfoundry.app');
                    $video_url = rtrim($wordfoundry_url, '/') . '/' . ltrim($video_url, '/');
                }

                // Upload video to WordPress Media Library
                $attachment_id = self::upload_media_from_url($video_url);

                if ($attachment_id) {
                    // Get the new WordPress URL
                    $new_url = wp_get_attachment_url($attachment_id);
                    if ($new_url) {
                        return '<video' . $before_src . 'src="' . esc_url($new_url) . '"' . $after_src . '>';
                    }
                }

                // If upload failed, keep original
                return $matches[0];
            },
            $content
        );

        return $content;
    }

    /**
     * Upload media (image/video) from URL to WordPress Media Library
     */
    private static function upload_media_from_url($url, $post_id = 0) {
        require_once(ABSPATH . 'wp-admin/includes/media.php');
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/image.php');

        // Download image
        $tmp = download_url($url);

        if (is_wp_error($tmp)) {
            return false;
        }

        // Get file info
        $file_array = array(
            'name' => basename(parse_url($url, PHP_URL_PATH)),
            'tmp_name' => $tmp,
        );

        // Upload to media library
        $attachment_id = media_handle_sideload($file_array, $post_id);

        // Clean up temp file
        @unlink($tmp);

        if (is_wp_error($attachment_id)) {
            return false;
        }

        return $attachment_id;
    }
}
