<?php
/**
 * WordFoundry Admin Settings Page
 */

if (!defined('ABSPATH')) {
    exit;
}

class WordFoundry_Admin {
    /**
     * Initialize admin hooks
     */
    public static function init() {
        add_action('admin_menu', array(__CLASS__, 'add_menu_page'));
        add_action('admin_init', array(__CLASS__, 'register_settings'));
        add_action('admin_enqueue_scripts', array(__CLASS__, 'enqueue_scripts'));
    }

    /**
     * Add menu page
     */
    public static function add_menu_page() {
        add_options_page(
            __('WordFoundry Settings', 'wordfoundry-connect'),
            __('WordFoundry', 'wordfoundry-connect'),
            'manage_options',
            'wordfoundry-settings',
            array(__CLASS__, 'render_settings_page')
        );
    }

    /**
     * Register settings
     */
    public static function register_settings() {
        register_setting(
            'wordfoundry_settings_group',
            'wordfoundry_settings',
            array(
                'sanitize_callback' => array(__CLASS__, 'sanitize_settings'),
            )
        );

        // Main section
        add_settings_section(
            'wordfoundry_main_section',
            __('Connection Settings', 'wordfoundry-connect'),
            array(__CLASS__, 'render_main_section'),
            'wordfoundry-settings'
        );

        // API URL field
        add_settings_field(
            'api_url',
            __('WordFoundry URL', 'wordfoundry-connect'),
            array(__CLASS__, 'render_api_url_field'),
            'wordfoundry-settings',
            'wordfoundry_main_section'
        );

        // Site ID field
        add_settings_field(
            'site_id',
            __('Site ID', 'wordfoundry-connect'),
            array(__CLASS__, 'render_site_id_field'),
            'wordfoundry-settings',
            'wordfoundry_main_section'
        );

        // Webhook Secret field
        add_settings_field(
            'webhook_secret',
            __('Webhook Secret', 'wordfoundry-connect'),
            array(__CLASS__, 'render_webhook_secret_field'),
            'wordfoundry-settings',
            'wordfoundry_main_section'
        );

        // Sync section
        add_settings_section(
            'wordfoundry_sync_section',
            __('Synchronization', 'wordfoundry-connect'),
            array(__CLASS__, 'render_sync_section'),
            'wordfoundry-settings'
        );

        // Two-way sync toggle
        add_settings_field(
            'sync_enabled',
            __('Two-Way Sync', 'wordfoundry-connect'),
            array(__CLASS__, 'render_sync_enabled_field'),
            'wordfoundry-settings',
            'wordfoundry_sync_section'
        );
    }

    /**
     * Sanitize settings
     */
    public static function sanitize_settings($input) {
        $sanitized = array();

        $sanitized['api_url'] = isset($input['api_url'])
            ? esc_url_raw($input['api_url'])
            : 'https://wordfoundry.app';

        $sanitized['site_id'] = isset($input['site_id'])
            ? sanitize_text_field($input['site_id'])
            : '';

        $sanitized['webhook_secret'] = isset($input['webhook_secret'])
            ? sanitize_text_field($input['webhook_secret'])
            : '';

        $sanitized['sync_enabled'] = isset($input['sync_enabled']) && $input['sync_enabled'] === '1';

        return $sanitized;
    }

    /**
     * Enqueue admin scripts
     */
    public static function enqueue_scripts($hook) {
        if ($hook !== 'settings_page_wordfoundry-settings') {
            return;
        }

        wp_enqueue_style(
            'wordfoundry-admin',
            WORDFOUNDRY_PLUGIN_URL . 'admin/css/admin.css',
            array(),
            WORDFOUNDRY_VERSION
        );
    }

    /**
     * Render settings page
     */
    public static function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }

        include WORDFOUNDRY_PLUGIN_DIR . 'admin/partials/settings-page.php';
    }

    /**
     * Section callbacks
     */
    public static function render_main_section() {
        echo '<p>' . esc_html__('Enter your WordFoundry connection details. You can find these in your WordFoundry dashboard under Site Settings.', 'wordfoundry-connect') . '</p>';
    }

    public static function render_sync_section() {
        echo '<p>' . esc_html__('Enable two-way synchronization to keep articles in sync between WordPress and WordFoundry.', 'wordfoundry-connect') . '</p>';
    }

    /**
     * Field callbacks
     */
    public static function render_api_url_field() {
        $settings = WordFoundry_Settings::get_all();
        ?>
        <input type="url"
               name="wordfoundry_settings[api_url]"
               value="<?php echo esc_attr($settings['api_url']); ?>"
               class="regular-text"
               placeholder="https://wordfoundry.app">
        <p class="description">
            <?php esc_html_e('The URL of your WordFoundry instance.', 'wordfoundry-connect'); ?>
        </p>
        <?php
    }

    public static function render_site_id_field() {
        $settings = WordFoundry_Settings::get_all();
        ?>
        <input type="text"
               name="wordfoundry_settings[site_id]"
               value="<?php echo esc_attr($settings['site_id']); ?>"
               class="regular-text"
               placeholder="<?php esc_attr_e('Your Site ID from WordFoundry', 'wordfoundry-connect'); ?>">
        <p class="description">
            <?php esc_html_e('Find this in WordFoundry under Sites > Your Site > Settings.', 'wordfoundry-connect'); ?>
        </p>
        <?php
    }

    public static function render_webhook_secret_field() {
        $settings = WordFoundry_Settings::get_all();
        ?>
        <input type="password"
               name="wordfoundry_settings[webhook_secret]"
               value="<?php echo esc_attr($settings['webhook_secret']); ?>"
               class="regular-text"
               placeholder="<?php esc_attr_e('Webhook Secret', 'wordfoundry-connect'); ?>">
        <p class="description">
            <?php esc_html_e('Secret key for verifying webhook signatures. Copy from WordFoundry site settings.', 'wordfoundry-connect'); ?>
        </p>
        <?php
    }

    public static function render_sync_enabled_field() {
        $settings = WordFoundry_Settings::get_all();
        ?>
        <label>
            <input type="checkbox"
                   name="wordfoundry_settings[sync_enabled]"
                   value="1"
                   <?php checked($settings['sync_enabled'], true); ?>>
            <?php esc_html_e('Enable two-way synchronization', 'wordfoundry-connect'); ?>
        </label>
        <p class="description">
            <?php esc_html_e('When enabled, changes made in WordPress will be synced back to WordFoundry.', 'wordfoundry-connect'); ?>
        </p>
        <?php
    }
}
