<?php

declare(strict_types=1);

namespace Drupal\wordfoundry_connect\Service;

use Drupal\node\NodeInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;
use Psr\Log\LoggerInterface;

/**
 * Service for synchronizing content with WordFoundry.
 *
 * Handles two-way sync between Drupal and WordFoundry:
 * - Sends webhooks to WordFoundry when nodes are updated/deleted in Drupal
 * - Prevents sync loops using timestamp flags
 */
class WordFoundrySync {

  /**
   * The HTTP client.
   */
  protected ClientInterface $httpClient;

  /**
   * The WordFoundry settings service.
   */
  protected WordFoundrySettings $settings;

  /**
   * The logger.
   */
  protected LoggerInterface $logger;

  /**
   * Constructs a new WordFoundrySync object.
   */
  public function __construct(
    ClientInterface $httpClient,
    WordFoundrySettings $settings,
    LoggerInterface $logger,
  ) {
    $this->httpClient = $httpClient;
    $this->settings = $settings;
    $this->logger = $logger;
  }

  /**
   * Check if we should sync this node.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node to check.
   *
   * @return bool
   *   TRUE if we should sync, FALSE otherwise.
   */
  public function shouldSync(NodeInterface $node): bool {
    // Check if sync is enabled.
    if (!$this->settings->get('sync_enabled', TRUE)) {
      return FALSE;
    }

    // Check if configured.
    if (!$this->settings->isConfigured()) {
      return FALSE;
    }

    // Check if this is the tracked content type.
    $trackedType = $this->settings->get('content_type', 'article');
    if ($node->bundle() !== $trackedType) {
      return FALSE;
    }

    // Check if this is a WordFoundry article.
    if (!$node->hasField('field_wordfoundry_id')) {
      return FALSE;
    }
    $wordfoundryId = $node->get('field_wordfoundry_id')->value;
    if (empty($wordfoundryId)) {
      return FALSE;
    }

    // Check sync flag to prevent loops.
    if ($node->hasField('field_wordfoundry_synced')) {
      $lastSynced = $node->get('field_wordfoundry_synced')->value;
      if (!empty($lastSynced)) {
        // This update came from WordFoundry, skip syncing back.
        return FALSE;
      }
    }

    return TRUE;
  }

  /**
   * Clear the sync flag after processing.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node to clear the flag on.
   */
  public function clearSyncFlag(NodeInterface $node): void {
    if ($node->hasField('field_wordfoundry_synced')) {
      $lastSynced = $node->get('field_wordfoundry_synced')->value;
      if (!empty($lastSynced)) {
        // Clear the flag without triggering another save.
        $connection = \Drupal::database();
        $connection->update('node__field_wordfoundry_synced')
          ->fields(['field_wordfoundry_synced_value' => NULL])
          ->condition('entity_id', $node->id())
          ->execute();
      }
    }
  }

  /**
   * Send node update to WordFoundry.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node that was updated.
   */
  public function sendUpdate(NodeInterface $node): void {
    $webhookUrl = $this->settings->getWebhookUrl();
    if (empty($webhookUrl)) {
      return;
    }

    $body = $node->hasField('body') ? $node->get('body')->value : '';

    $payload = [
      'event' => 'post.updated',
      'post_id' => (int) $node->id(),
      'title' => $node->getTitle(),
      'content' => $body,
      'status' => $node->isPublished() ? 'publish' : 'draft',
      'modified_at' => date('c', $node->getChangedTime()),
    ];

    $this->sendWebhook($webhookUrl, $payload);
  }

  /**
   * Send node delete notification to WordFoundry.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node that was deleted.
   */
  public function sendDelete(NodeInterface $node): void {
    $webhookUrl = $this->settings->getWebhookUrl();
    if (empty($webhookUrl)) {
      return;
    }

    $articleId = $node->hasField('field_wordfoundry_id')
      ? $node->get('field_wordfoundry_id')->value
      : NULL;

    if (empty($articleId)) {
      return;
    }

    $payload = [
      'event' => 'post.deleted',
      'post_id' => (int) $node->id(),
      'article_id' => (int) $articleId,
      'deleted_at' => date('c'),
    ];

    $this->sendWebhook($webhookUrl, $payload);
  }

  /**
   * Send node trash notification to WordFoundry.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node that was trashed (unpublished).
   */
  public function sendTrash(NodeInterface $node): void {
    $webhookUrl = $this->settings->getWebhookUrl();
    if (empty($webhookUrl)) {
      return;
    }

    $articleId = $node->hasField('field_wordfoundry_id')
      ? $node->get('field_wordfoundry_id')->value
      : NULL;

    if (empty($articleId)) {
      return;
    }

    $payload = [
      'event' => 'post.trashed',
      'post_id' => (int) $node->id(),
      'article_id' => (int) $articleId,
      'timestamp' => date('c'),
    ];

    $this->sendWebhook($webhookUrl, $payload);
  }

  /**
   * Send node restore notification to WordFoundry.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node that was restored (published).
   */
  public function sendRestore(NodeInterface $node): void {
    $webhookUrl = $this->settings->getWebhookUrl();
    if (empty($webhookUrl)) {
      return;
    }

    $articleId = $node->hasField('field_wordfoundry_id')
      ? $node->get('field_wordfoundry_id')->value
      : NULL;

    if (empty($articleId)) {
      return;
    }

    $payload = [
      'event' => 'post.restored',
      'post_id' => (int) $node->id(),
      'article_id' => (int) $articleId,
      'status' => $node->isPublished() ? 'publish' : 'draft',
      'timestamp' => date('c'),
    ];

    $this->sendWebhook($webhookUrl, $payload);
  }

  /**
   * Send webhook to WordFoundry.
   *
   * @param string $url
   *   The webhook URL.
   * @param array $payload
   *   The payload to send.
   */
  protected function sendWebhook(string $url, array $payload): void {
    $body = json_encode($payload);

    // Create HMAC signature.
    $webhookSecret = $this->settings->get('webhook_secret', '');
    $signature = '';
    if (!empty($webhookSecret)) {
      $signature = 'sha256=' . hash_hmac('sha256', $body, $webhookSecret);
    }

    try {
      $response = $this->httpClient->request('POST', $url, [
        'headers' => [
          'Content-Type' => 'application/json',
          'X-WF-Signature' => $signature,
        ],
        'body' => $body,
        'timeout' => 30,
      ]);

      $statusCode = $response->getStatusCode();
      if ($statusCode >= 400) {
        $this->logger->error('WordFoundry webhook @event failed: HTTP @code', [
          '@event' => $payload['event'] ?? 'unknown',
          '@code' => $statusCode,
        ]);
      }
      else {
        $this->logger->info('WordFoundry webhook @event sent successfully', [
          '@event' => $payload['event'] ?? 'unknown',
        ]);
      }
    }
    catch (GuzzleException $e) {
      $this->logger->error('WordFoundry webhook @event error: @message', [
        '@event' => $payload['event'] ?? 'unknown',
        '@message' => $e->getMessage(),
      ]);
    }
  }

  /**
   * Manual sync trigger.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node to sync.
   *
   * @return bool
   *   TRUE on success, FALSE on failure.
   */
  public function manualSync(NodeInterface $node): bool {
    if (!$this->settings->isConfigured()) {
      return FALSE;
    }

    $this->sendUpdate($node);
    return TRUE;
  }

}
