<?php

declare(strict_types=1);

namespace Drupal\wordfoundry_connect\Service;

use Drupal\Core\Config\ConfigFactoryInterface;

/**
 * Service for managing WordFoundry settings.
 */
class WordFoundrySettings {

  /**
   * The configuration factory.
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Default settings values.
   */
  protected const DEFAULTS = [
    'api_url' => 'https://wordfoundry.app',
    'api_key' => '',
    'site_id' => '',
    'webhook_secret' => '',
    'sync_enabled' => TRUE,
    'content_type' => 'article',
    'taxonomy' => 'tags',
  ];

  /**
   * Constructs a new WordFoundrySettings object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->configFactory = $config_factory;
  }

  /**
   * Gets all settings.
   *
   * @return array
   *   All settings with defaults applied.
   */
  public function getAll(): array {
    $config = $this->configFactory->get('wordfoundry_connect.settings');
    $settings = [];

    foreach (self::DEFAULTS as $key => $default) {
      $value = $config->get($key);
      $settings[$key] = $value ?? $default;
    }

    return $settings;
  }

  /**
   * Gets a single setting value.
   *
   * @param string $key
   *   The setting key.
   * @param mixed $default
   *   Default value if not set.
   *
   * @return mixed
   *   The setting value.
   */
  public function get(string $key, mixed $default = NULL): mixed {
    $config = $this->configFactory->get('wordfoundry_connect.settings');
    $value = $config->get($key);

    if ($value === NULL) {
      return $default ?? (self::DEFAULTS[$key] ?? NULL);
    }

    return $value;
  }

  /**
   * Sets a single setting value.
   *
   * @param string $key
   *   The setting key.
   * @param mixed $value
   *   The value to set.
   */
  public function set(string $key, mixed $value): void {
    $this->configFactory->getEditable('wordfoundry_connect.settings')
      ->set($key, $value)
      ->save();
  }

  /**
   * Updates multiple settings at once.
   *
   * @param array $values
   *   An associative array of settings to update.
   */
  public function update(array $values): void {
    $config = $this->configFactory->getEditable('wordfoundry_connect.settings');

    foreach ($values as $key => $value) {
      $config->set($key, $value);
    }

    $config->save();
  }

  /**
   * Checks if the module is properly configured.
   *
   * @return bool
   *   TRUE if configured, FALSE otherwise.
   */
  public function isConfigured(): bool {
    $api_key = $this->get('api_key');
    $site_id = $this->get('site_id');
    $webhook_secret = $this->get('webhook_secret');

    return !empty($api_key) && !empty($site_id) && !empty($webhook_secret);
  }

  /**
   * Gets the webhook URL for this site.
   *
   * @return string|null
   *   The webhook URL or NULL if not configured.
   */
  public function getWebhookUrl(): ?string {
    $api_url = $this->get('api_url');
    $site_id = $this->get('site_id');

    if (empty($api_url) || empty($site_id)) {
      return NULL;
    }

    return rtrim($api_url, '/') . '/api/v1/sites/webhook/' . $site_id;
  }

  /**
   * Validates the API key from request.
   *
   * @param string $api_key
   *   The API key from request.
   *
   * @return bool
   *   TRUE if valid, FALSE otherwise.
   */
  public function validateApiKey(string $api_key): bool {
    $stored_key = $this->get('api_key');

    if (empty($stored_key) || empty($api_key)) {
      return FALSE;
    }

    return hash_equals($stored_key, $api_key);
  }

}
