<?php

declare(strict_types=1);

namespace Drupal\wordfoundry_connect\Service;

use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\file\Entity\File;
use Drupal\file\FileInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\GuzzleException;
use Psr\Log\LoggerInterface;

/**
 * Service for handling media downloads and processing.
 */
class MediaHandler {

  /**
   * The HTTP client.
   */
  protected ClientInterface $httpClient;

  /**
   * The file system service.
   */
  protected FileSystemInterface $fileSystem;

  /**
   * The file URL generator.
   */
  protected FileUrlGeneratorInterface $fileUrlGenerator;

  /**
   * The WordFoundry settings service.
   */
  protected WordFoundrySettings $settings;

  /**
   * The logger.
   */
  protected LoggerInterface $logger;

  /**
   * Constructs a new MediaHandler object.
   */
  public function __construct(
    ClientInterface $httpClient,
    FileSystemInterface $fileSystem,
    FileUrlGeneratorInterface $fileUrlGenerator,
    WordFoundrySettings $settings,
    LoggerInterface $logger,
  ) {
    $this->httpClient = $httpClient;
    $this->fileSystem = $fileSystem;
    $this->fileUrlGenerator = $fileUrlGenerator;
    $this->settings = $settings;
    $this->logger = $logger;
  }

  /**
   * Download and save an image from URL.
   *
   * @param string $url
   *   The URL of the image to download.
   * @param string|null $filename
   *   Optional filename override.
   *
   * @return \Drupal\file\FileInterface|null
   *   The saved file entity or NULL on failure.
   */
  public function downloadAndSaveImage(string $url, ?string $filename = NULL): ?FileInterface {
    try {
      // Download the image.
      $response = $this->httpClient->request('GET', $url, [
        'timeout' => 30,
        'headers' => [
          'User-Agent' => 'WordFoundry-Drupal/1.0',
        ],
      ]);

      if ($response->getStatusCode() !== 200) {
        $this->logger->warning('Failed to download image from @url: HTTP @code', [
          '@url' => $url,
          '@code' => $response->getStatusCode(),
        ]);
        return NULL;
      }

      $data = $response->getBody()->getContents();
      if (empty($data)) {
        $this->logger->warning('Empty response when downloading image from @url', [
          '@url' => $url,
        ]);
        return NULL;
      }

      // Determine filename.
      if (empty($filename)) {
        $parsedUrl = parse_url($url);
        $filename = basename($parsedUrl['path'] ?? 'image.jpg');
        // Clean up filename.
        $filename = preg_replace('/[^a-zA-Z0-9._-]/', '', $filename);
        if (empty($filename) || strpos($filename, '.') === FALSE) {
          $filename = 'image_' . time() . '.jpg';
        }
      }

      // Ensure the directory exists.
      $directory = 'public://wordfoundry';
      $this->fileSystem->prepareDirectory($directory, FileSystemInterface::CREATE_DIRECTORY);

      // Generate unique filename.
      $destination = $directory . '/' . $filename;
      $destination = $this->fileSystem->getDestinationFilename($destination, FileSystemInterface::EXISTS_RENAME);

      // Save the file.
      $uri = $this->fileSystem->saveData($data, $destination);
      if (!$uri) {
        $this->logger->error('Failed to save image data to @destination', [
          '@destination' => $destination,
        ]);
        return NULL;
      }

      // Create file entity.
      $file = File::create([
        'uri' => $uri,
        'filename' => basename($uri),
        'status' => 1,
      ]);
      $file->save();

      $this->logger->info('Downloaded and saved image from @url to @uri', [
        '@url' => $url,
        '@uri' => $uri,
      ]);

      return $file;
    }
    catch (GuzzleException $e) {
      $this->logger->error('Error downloading image from @url: @message', [
        '@url' => $url,
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
    catch (\Exception $e) {
      $this->logger->error('Error processing image from @url: @message', [
        '@url' => $url,
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Process content - download external images and replace URLs.
   *
   * @param string $content
   *   The HTML content to process.
   *
   * @return string
   *   The processed content with local image URLs.
   */
  public function processContent(string $content): string {
    // First convert markdown images to HTML.
    // Pattern: ![alt text](url)
    $content = preg_replace_callback(
      '/!\[([^\]]*)\]\(([^)]+)\)/',
      function ($matches) {
        $alt = htmlspecialchars($matches[1], ENT_QUOTES, 'UTF-8');
        $url = htmlspecialchars($matches[2], ENT_QUOTES, 'UTF-8');
        return '<img src="' . $url . '" alt="' . $alt . '" />';
      },
      $content
    );

    // Get base URL for comparison.
    $baseUrl = \Drupal::request()->getSchemeAndHttpHost();

    // Process all img tags and upload images.
    $content = preg_replace_callback(
      '/<img([^>]*)src=["\']([^"\']+)["\']([^>]*)>/i',
      function ($matches) use ($baseUrl) {
        $beforeSrc = $matches[1];
        $imageUrl = $matches[2];
        $afterSrc = $matches[3];

        // Skip if already a local URL or data URL.
        if (str_starts_with($imageUrl, $baseUrl) || str_starts_with($imageUrl, 'data:')) {
          return $matches[0];
        }

        // Make URL absolute if relative.
        if (!str_starts_with($imageUrl, 'http')) {
          $wordfoundryUrl = $this->settings->get('api_url', 'https://wordfoundry.app');
          $imageUrl = rtrim($wordfoundryUrl, '/') . '/' . ltrim($imageUrl, '/');
        }

        // Download and save the image.
        $file = $this->downloadAndSaveImage($imageUrl);
        if ($file) {
          $newUrl = $this->fileUrlGenerator->generateAbsoluteString($file->getFileUri());
          return '<img' . $beforeSrc . 'src="' . htmlspecialchars($newUrl, ENT_QUOTES, 'UTF-8') . '"' . $afterSrc . '>';
        }

        // If download failed, keep original.
        return $matches[0];
      },
      $content
    );

    // Process video tags.
    $content = preg_replace_callback(
      '/<video([^>]*)src=["\']([^"\']+)["\']([^>]*)>/i',
      function ($matches) use ($baseUrl) {
        $beforeSrc = $matches[1];
        $videoUrl = $matches[2];
        $afterSrc = $matches[3];

        // Skip if already a local URL.
        if (str_starts_with($videoUrl, $baseUrl)) {
          return $matches[0];
        }

        // Make URL absolute if relative.
        if (!str_starts_with($videoUrl, 'http')) {
          $wordfoundryUrl = $this->settings->get('api_url', 'https://wordfoundry.app');
          $videoUrl = rtrim($wordfoundryUrl, '/') . '/' . ltrim($videoUrl, '/');
        }

        // Download and save the video.
        $file = $this->downloadAndSaveVideo($videoUrl);
        if ($file) {
          $newUrl = $this->fileUrlGenerator->generateAbsoluteString($file->getFileUri());
          return '<video' . $beforeSrc . 'src="' . htmlspecialchars($newUrl, ENT_QUOTES, 'UTF-8') . '"' . $afterSrc . '>';
        }

        // If download failed, keep original.
        return $matches[0];
      },
      $content
    );

    return $content;
  }

  /**
   * Download and save a video from URL.
   *
   * @param string $url
   *   The URL of the video to download.
   *
   * @return \Drupal\file\FileInterface|null
   *   The saved file entity or NULL on failure.
   */
  public function downloadAndSaveVideo(string $url): ?FileInterface {
    try {
      // Download the video.
      $response = $this->httpClient->request('GET', $url, [
        'timeout' => 120,
        'headers' => [
          'User-Agent' => 'WordFoundry-Drupal/1.0',
        ],
      ]);

      if ($response->getStatusCode() !== 200) {
        $this->logger->warning('Failed to download video from @url: HTTP @code', [
          '@url' => $url,
          '@code' => $response->getStatusCode(),
        ]);
        return NULL;
      }

      $data = $response->getBody()->getContents();
      if (empty($data)) {
        return NULL;
      }

      // Determine filename.
      $parsedUrl = parse_url($url);
      $filename = basename($parsedUrl['path'] ?? 'video.mp4');
      $filename = preg_replace('/[^a-zA-Z0-9._-]/', '', $filename);
      if (empty($filename) || strpos($filename, '.') === FALSE) {
        $filename = 'video_' . time() . '.mp4';
      }

      // Ensure the directory exists.
      $directory = 'public://wordfoundry/videos';
      $this->fileSystem->prepareDirectory($directory, FileSystemInterface::CREATE_DIRECTORY);

      // Generate unique filename.
      $destination = $directory . '/' . $filename;
      $destination = $this->fileSystem->getDestinationFilename($destination, FileSystemInterface::EXISTS_RENAME);

      // Save the file.
      $uri = $this->fileSystem->saveData($data, $destination);
      if (!$uri) {
        return NULL;
      }

      // Create file entity.
      $file = File::create([
        'uri' => $uri,
        'filename' => basename($uri),
        'status' => 1,
      ]);
      $file->save();

      return $file;
    }
    catch (\Exception $e) {
      $this->logger->error('Error downloading video from @url: @message', [
        '@url' => $url,
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

}
