<?php

declare(strict_types=1);

namespace Drupal\wordfoundry_connect\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\node\Entity\NodeType;
use Drupal\taxonomy\Entity\Vocabulary;
use Drupal\wordfoundry_connect\Service\WordFoundrySettings;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure WordFoundry Connect settings.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * The WordFoundry settings service.
   */
  protected WordFoundrySettings $settings;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    $instance = parent::create($container);
    $instance->settings = $container->get('wordfoundry_connect.settings');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'wordfoundry_connect_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['wordfoundry_connect.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $is_configured = $this->settings->isConfigured();

    // Status message.
    if (!$is_configured) {
      $this->messenger()->addWarning($this->t('Setup Required: Please enter your API Key, Site ID and Webhook Secret to connect to WordFoundry.'));
    }
    else {
      $this->messenger()->addStatus($this->t('Your site is connected to WordFoundry.'));
    }

    // Connection settings.
    $form['connection'] = [
      '#type' => 'details',
      '#title' => $this->t('Connection Settings'),
      '#open' => TRUE,
    ];

    $form['connection']['api_url'] = [
      '#type' => 'url',
      '#title' => $this->t('WordFoundry API URL'),
      '#description' => $this->t('The WordFoundry API server URL.'),
      '#default_value' => $this->settings->get('api_url'),
      '#required' => TRUE,
    ];

    $form['connection']['api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API Key'),
      '#description' => $this->t('The API key for authenticating requests from WordFoundry. Generate this in your WordFoundry account.'),
      '#default_value' => $this->settings->get('api_key'),
      '#required' => TRUE,
      '#attributes' => [
        'autocomplete' => 'off',
      ],
    ];

    $form['connection']['site_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Site ID'),
      '#description' => $this->t('Your unique Site ID from WordFoundry. Find this in your site settings.'),
      '#default_value' => $this->settings->get('site_id'),
      '#required' => TRUE,
    ];

    $form['connection']['webhook_secret'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Webhook Secret'),
      '#description' => $this->t('Secret key for verifying webhook signatures from WordFoundry.'),
      '#default_value' => $this->settings->get('webhook_secret'),
      '#required' => TRUE,
      '#attributes' => [
        'autocomplete' => 'off',
      ],
    ];

    // Sync settings.
    $form['sync'] = [
      '#type' => 'details',
      '#title' => $this->t('Synchronization Settings'),
      '#open' => TRUE,
    ];

    $form['sync']['sync_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable two-way synchronization'),
      '#description' => $this->t('When enabled, changes to content in Drupal will be sent back to WordFoundry.'),
      '#default_value' => $this->settings->get('sync_enabled'),
    ];

    // Content type options.
    $content_types = [];
    foreach (NodeType::loadMultiple() as $type) {
      $content_types[$type->id()] = $type->label();
    }

    $form['sync']['content_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Default Content Type'),
      '#description' => $this->t('The default content type for published articles.'),
      '#options' => $content_types,
      '#default_value' => $this->settings->get('content_type'),
      '#required' => TRUE,
    ];

    // Taxonomy options.
    $taxonomies = [];
    foreach (Vocabulary::loadMultiple() as $vocab) {
      $taxonomies[$vocab->id()] = $vocab->label();
    }

    $form['sync']['taxonomy'] = [
      '#type' => 'select',
      '#title' => $this->t('Default Taxonomy'),
      '#description' => $this->t('The default taxonomy vocabulary for article tags/categories.'),
      '#options' => $taxonomies,
      '#default_value' => $this->settings->get('taxonomy'),
      '#empty_option' => $this->t('- None -'),
    ];

    // Information sidebar.
    $form['info'] = [
      '#type' => 'details',
      '#title' => $this->t('Setup Information'),
      '#open' => TRUE,
    ];

    $form['info']['quick_setup'] = [
      '#type' => 'item',
      '#title' => $this->t('Quick Setup'),
      '#markup' => '<ol>
        <li>' . $this->t('Log in to WordFoundry') . '</li>
        <li>' . $this->t('Go to Sites and add your Drupal site') . '</li>
        <li>' . $this->t('Copy the API Key, Site ID and Webhook Secret') . '</li>
        <li>' . $this->t('Paste them in the fields above') . '</li>
        <li>' . $this->t('Save settings and start publishing!') . '</li>
      </ol>',
    ];

    // API Endpoint info.
    $api_endpoint = $this->getRequest()->getSchemeAndHttpHost() . '/api/wordfoundry/v1/';
    $form['info']['api_endpoint'] = [
      '#type' => 'item',
      '#title' => $this->t('REST API Endpoint'),
      '#markup' => '<code style="display: block; padding: 10px; background: #f1f1f1; word-break: break-all;">' . $api_endpoint . '</code>',
    ];

    // Webhook URL info.
    if ($is_configured) {
      $webhook_url = $this->settings->getWebhookUrl();
      $form['info']['webhook_url'] = [
        '#type' => 'item',
        '#title' => $this->t('Webhook URL'),
        '#markup' => '<p>' . $this->t('WordFoundry will send updates to:') . '</p>
          <code style="display: block; padding: 10px; background: #f1f1f1; word-break: break-all;">' . $webhook_url . '</code>',
      ];
    }

    // Help links.
    $form['info']['help_links'] = [
      '#type' => 'item',
      '#title' => $this->t('Need Help?'),
      '#markup' => '<p><a href="https://wordfoundry.app/docs" target="_blank" rel="noopener">' . $this->t('Documentation') . '</a></p>
        <p><a href="https://wordfoundry.app/support" target="_blank" rel="noopener">' . $this->t('Contact Support') . '</a></p>',
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);

    // Validate API URL format.
    $api_url = $form_state->getValue('api_url');
    if (!filter_var($api_url, FILTER_VALIDATE_URL)) {
      $form_state->setErrorByName('api_url', $this->t('Please enter a valid URL.'));
    }

    // Validate Site ID format (should be UUID-like or alphanumeric).
    $site_id = $form_state->getValue('site_id');
    if (!empty($site_id) && !preg_match('/^[a-zA-Z0-9\-_]+$/', $site_id)) {
      $form_state->setErrorByName('site_id', $this->t('Site ID should contain only letters, numbers, hyphens and underscores.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->settings->update([
      'api_url' => rtrim($form_state->getValue('api_url'), '/'),
      'api_key' => $form_state->getValue('api_key'),
      'site_id' => $form_state->getValue('site_id'),
      'webhook_secret' => $form_state->getValue('webhook_secret'),
      'sync_enabled' => (bool) $form_state->getValue('sync_enabled'),
      'content_type' => $form_state->getValue('content_type'),
      'taxonomy' => $form_state->getValue('taxonomy'),
    ]);

    parent::submitForm($form, $form_state);
  }

}
