<?php

declare(strict_types=1);

namespace Drupal\wordfoundry_connect\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\taxonomy\Entity\Term;
use Drupal\taxonomy\Entity\Vocabulary;
use Drupal\wordfoundry_connect\Service\MediaHandler;
use Drupal\wordfoundry_connect\Service\WordFoundrySettings;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * API Controller for WordFoundry integration.
 *
 * Provides REST endpoints for WordFoundry to interact with Drupal:
 * - POST /api/wordfoundry/v1/verify - Verify connection
 * - GET /api/wordfoundry/v1/settings - Get site settings
 * - GET /api/wordfoundry/v1/content-types - Get available content types
 * - GET /api/wordfoundry/v1/taxonomies/{content_type} - Get taxonomies for type
 * - GET /api/wordfoundry/v1/terms/{taxonomy} - Get terms for taxonomy
 * - POST /api/wordfoundry/v1/publish - Publish a node
 * - PUT /api/wordfoundry/v1/nodes/{id} - Update a node
 * - DELETE /api/wordfoundry/v1/nodes/{id} - Delete a node
 * - POST /api/wordfoundry/v1/nodes/{id}/restore - Restore a node
 * - GET /api/wordfoundry/v1/nodes - List nodes for import
 * - GET /api/wordfoundry/v1/nodes/{id}/export - Export a node
 * - GET /api/wordfoundry/v1/nodes/similar - Find similar nodes
 */
class ApiController extends ControllerBase {

  /**
   * Plugin version.
   */
  const VERSION = '1.0.0';

  /**
   * The WordFoundry settings service.
   */
  protected WordFoundrySettings $settings;

  /**
   * The media handler service.
   */
  protected MediaHandler $mediaHandler;

  /**
   * The entity type manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    $instance = parent::create($container);
    $instance->settings = $container->get('wordfoundry_connect.settings');
    $instance->mediaHandler = $container->get('wordfoundry_connect.media_handler');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * Authenticate API request.
   */
  protected function authenticate(Request $request): bool {
    $apiKey = $request->headers->get('X-API-Key');
    if (empty($apiKey)) {
      return FALSE;
    }
    return $this->settings->validateApiKey($apiKey);
  }

  /**
   * Return unauthorized response.
   */
  protected function unauthorized(): JsonResponse {
    return new JsonResponse([
      'success' => FALSE,
      'error' => [
        'code' => 'UNAUTHORIZED',
        'message' => 'Invalid or missing API key',
      ],
    ], 401);
  }

  /**
   * POST /api/wordfoundry/v1/verify - Verify connection.
   */
  public function verify(Request $request): JsonResponse {
    if (!$this->authenticate($request)) {
      return $this->unauthorized();
    }

    return new JsonResponse([
      'success' => TRUE,
      'site_name' => $this->config('system.site')->get('name'),
      'message' => 'Connection verified successfully',
      'drupal_version' => \Drupal::VERSION,
      'plugin_version' => self::VERSION,
    ]);
  }

  /**
   * GET /api/wordfoundry/v1/settings - Get site settings.
   */
  public function getSettings(Request $request): JsonResponse {
    if (!$this->authenticate($request)) {
      return $this->unauthorized();
    }

    $settings = $this->settings->getAll();

    // Get content types.
    $contentTypes = [];
    foreach (NodeType::loadMultiple() as $type) {
      $contentTypes[] = [
        'name' => $type->id(),
        'label' => $type->label(),
      ];
    }

    // Get default taxonomy terms.
    $taxonomy = $settings['taxonomy'] ?? 'tags';
    $terms = [];
    if ($taxonomy && $this->entityTypeManager->getStorage('taxonomy_vocabulary')->load($taxonomy)) {
      $termEntities = $this->entityTypeManager
        ->getStorage('taxonomy_term')
        ->loadByProperties(['vid' => $taxonomy]);
      foreach ($termEntities as $term) {
        $terms[] = [
          'id' => $term->id(),
          'name' => $term->getName(),
          'slug' => $term->id(),
        ];
      }
    }

    return new JsonResponse([
      'site_name' => $this->config('system.site')->get('name'),
      'cms' => 'drupal',
      'cms_version' => \Drupal::VERSION,
      'plugin_version' => self::VERSION,
      'post_types' => $contentTypes,
      'categories' => $terms,
      'sync_enabled' => $settings['sync_enabled'],
    ]);
  }

  /**
   * GET /api/wordfoundry/v1/content-types - Get available content types.
   */
  public function getContentTypes(Request $request): JsonResponse {
    if (!$this->authenticate($request)) {
      return $this->unauthorized();
    }

    $result = [];
    foreach (NodeType::loadMultiple() as $type) {
      $result[] = [
        'name' => $type->id(),
        'label' => $type->label(),
        'label_plural' => $type->label(),
        'description' => $type->getDescription(),
      ];
    }

    return new JsonResponse([
      'success' => TRUE,
      'post_types' => $result,
    ]);
  }

  /**
   * GET /api/wordfoundry/v1/taxonomies/{content_type} - Get taxonomies for content type.
   */
  public function getTaxonomies(string $content_type, Request $request): JsonResponse {
    if (!$this->authenticate($request)) {
      return $this->unauthorized();
    }

    // Check if content type exists.
    $nodeType = NodeType::load($content_type);
    if (!$nodeType) {
      return new JsonResponse([
        'success' => FALSE,
        'error' => [
          'code' => 'INVALID_CONTENT_TYPE',
          'message' => 'Content type does not exist',
        ],
      ], 400);
    }

    // Get all vocabulary (taxonomies) as Drupal doesn't directly tie them to content types.
    $result = [];
    foreach (Vocabulary::loadMultiple() as $vocab) {
      $result[] = [
        'name' => $vocab->id(),
        'label' => $vocab->label(),
        'label_plural' => $vocab->label(),
        'hierarchical' => TRUE,
        'show_ui' => TRUE,
      ];
    }

    return new JsonResponse([
      'success' => TRUE,
      'post_type' => $content_type,
      'taxonomies' => $result,
    ]);
  }

  /**
   * GET /api/wordfoundry/v1/terms/{taxonomy} - Get terms for taxonomy.
   */
  public function getTerms(string $taxonomy, Request $request): JsonResponse {
    if (!$this->authenticate($request)) {
      return $this->unauthorized();
    }

    $perPage = min((int) $request->query->get('per_page', 100), 500);
    $search = $request->query->get('search', '');

    // Check if vocabulary exists.
    $vocab = Vocabulary::load($taxonomy);
    if (!$vocab) {
      return new JsonResponse([
        'success' => FALSE,
        'error' => [
          'code' => 'INVALID_TAXONOMY',
          'message' => 'Taxonomy does not exist',
        ],
      ], 400);
    }

    // Load terms.
    $query = $this->entityTypeManager
      ->getStorage('taxonomy_term')
      ->getQuery()
      ->accessCheck(FALSE)
      ->condition('vid', $taxonomy)
      ->range(0, $perPage);

    if (!empty($search)) {
      $query->condition('name', '%' . $search . '%', 'LIKE');
    }

    $termIds = $query->execute();
    $terms = Term::loadMultiple($termIds);

    $result = [];
    foreach ($terms as $term) {
      $result[] = [
        'id' => (int) $term->id(),
        'name' => $term->getName(),
        'slug' => $term->id(),
        'count' => 0,
        'parent' => (int) ($term->get('parent')->target_id ?? 0),
      ];
    }

    return new JsonResponse([
      'success' => TRUE,
      'taxonomy' => $taxonomy,
      'taxonomy_label' => $vocab->label(),
      'terms' => $result,
      'total' => count($result),
    ]);
  }

  /**
   * POST /api/wordfoundry/v1/publish - Publish a node.
   */
  public function publish(Request $request): JsonResponse {
    if (!$this->authenticate($request)) {
      return $this->unauthorized();
    }

    $params = json_decode($request->getContent(), TRUE) ?? [];

    // Required fields.
    if (empty($params['title']) || empty($params['content'])) {
      return new JsonResponse([
        'success' => FALSE,
        'error' => [
          'code' => 'MISSING_FIELDS',
          'message' => 'Title and content are required',
        ],
      ], 400);
    }

    // Get content type.
    $contentType = $params['post_type'] ?? $this->settings->get('content_type', 'article');
    if (!NodeType::load($contentType)) {
      return new JsonResponse([
        'success' => FALSE,
        'error' => [
          'code' => 'INVALID_CONTENT_TYPE',
          'message' => 'Content type does not exist',
        ],
      ], 400);
    }

    // Process content - download images and replace URLs.
    $processedContent = $this->mediaHandler->processContent($params['content']);

    // Determine status.
    $status = match ($params['status'] ?? 'draft') {
      'publish', 'published' => TRUE,
      default => FALSE,
    };

    // Create node.
    $nodeData = [
      'type' => $contentType,
      'title' => $params['title'],
      'body' => [
        'value' => $processedContent,
        'format' => 'full_html',
      ],
      'status' => $status,
    ];

    // Set author if provided.
    if (!empty($params['author']) && is_numeric($params['author'])) {
      $user = $this->entityTypeManager->getStorage('user')->load($params['author']);
      if ($user) {
        $nodeData['uid'] = $params['author'];
      }
    }

    $node = Node::create($nodeData);

    // Set WordFoundry article ID.
    if (!empty($params['article_id']) && $node->hasField('field_wordfoundry_id')) {
      $node->set('field_wordfoundry_id', (string) $params['article_id']);
    }

    // Handle taxonomy terms.
    if (!empty($params['terms']) && is_array($params['terms'])) {
      foreach ($params['terms'] as $termData) {
        if (!empty($termData['taxonomy']) && !empty($termData['term_ids'])) {
          $fieldName = $this->getTaxonomyFieldName($contentType, $termData['taxonomy']);
          if ($fieldName && $node->hasField($fieldName)) {
            $termIds = array_map('intval', (array) $termData['term_ids']);
            $node->set($fieldName, $termIds);
          }
        }
      }
    }
    // Backwards compatibility: categories as simple array.
    elseif (!empty($params['categories'])) {
      $taxonomy = $params['taxonomy'] ?? $this->settings->get('taxonomy', 'tags');
      $fieldName = $this->getTaxonomyFieldName($contentType, $taxonomy);
      if ($fieldName && $node->hasField($fieldName)) {
        $termIds = array_map('intval', (array) $params['categories']);
        $node->set($fieldName, $termIds);
      }
    }

    // Handle featured image.
    if (!empty($params['featured_image_url'])) {
      $fileEntity = $this->mediaHandler->downloadAndSaveImage($params['featured_image_url']);
      if ($fileEntity && $node->hasField('field_image')) {
        $node->set('field_image', [
          'target_id' => $fileEntity->id(),
          'alt' => $params['title'],
        ]);
      }
    }

    // Store FAQ Schema.org JSON-LD.
    if (!empty($params['faq_schema']) && $node->hasField('field_schema_faq')) {
      $node->set('field_schema_faq', $params['faq_schema']);
    }

    // Mark as synced from WordFoundry (to avoid sync loop).
    if ($node->hasField('field_wordfoundry_synced')) {
      $node->set('field_wordfoundry_synced', time());
    }

    try {
      $node->save();
    }
    catch (\Exception $e) {
      return new JsonResponse([
        'success' => FALSE,
        'error' => [
          'code' => 'CREATE_FAILED',
          'message' => $e->getMessage(),
        ],
      ], 500);
    }

    return new JsonResponse([
      'success' => TRUE,
      'post_id' => (int) $node->id(),
      'permalink' => $node->toUrl()->setAbsolute()->toString(),
      'status' => $node->isPublished() ? 'publish' : 'draft',
    ], 201);
  }

  /**
   * PUT /api/wordfoundry/v1/nodes/{id} - Update a node.
   */
  public function updateNode(int $id, Request $request): JsonResponse {
    if (!$this->authenticate($request)) {
      return $this->unauthorized();
    }

    $node = Node::load($id);
    if (!$node) {
      return new JsonResponse([
        'success' => FALSE,
        'error' => [
          'code' => 'NOT_FOUND',
          'message' => 'Node not found',
        ],
      ], 404);
    }

    $params = json_decode($request->getContent(), TRUE) ?? [];

    if (isset($params['title'])) {
      $node->setTitle($params['title']);
    }

    if (isset($params['content'])) {
      $processedContent = $this->mediaHandler->processContent($params['content']);
      $node->set('body', [
        'value' => $processedContent,
        'format' => 'full_html',
      ]);
    }

    if (isset($params['status'])) {
      $status = match ($params['status']) {
        'publish', 'published' => TRUE,
        default => FALSE,
      };
      $node->setPublished($status);
    }

    // Mark as synced from WordFoundry (to avoid sync loop).
    if ($node->hasField('field_wordfoundry_synced')) {
      $node->set('field_wordfoundry_synced', time());
    }

    try {
      $node->save();
    }
    catch (\Exception $e) {
      return new JsonResponse([
        'success' => FALSE,
        'error' => [
          'code' => 'UPDATE_FAILED',
          'message' => $e->getMessage(),
        ],
      ], 500);
    }

    return new JsonResponse([
      'success' => TRUE,
      'post_id' => (int) $node->id(),
      'permalink' => $node->toUrl()->setAbsolute()->toString(),
      'updated_at' => date('c', $node->getChangedTime()),
    ]);
  }

  /**
   * DELETE /api/wordfoundry/v1/nodes/{id} - Delete a node.
   */
  public function deleteNode(int $id, Request $request): JsonResponse {
    if (!$this->authenticate($request)) {
      return $this->unauthorized();
    }

    $node = Node::load($id);
    if (!$node) {
      return new JsonResponse([
        'success' => FALSE,
        'error' => [
          'code' => 'NOT_FOUND',
          'message' => 'Node not found',
        ],
      ], 404);
    }

    // Drupal doesn't have a trash system by default.
    // We'll unpublish instead of deleting.
    $node->setUnpublished();

    try {
      $node->save();
    }
    catch (\Exception $e) {
      return new JsonResponse([
        'success' => FALSE,
        'error' => [
          'code' => 'DELETE_FAILED',
          'message' => $e->getMessage(),
        ],
      ], 500);
    }

    return new JsonResponse([
      'success' => TRUE,
      'post_id' => (int) $node->id(),
      'status' => 'trashed',
    ]);
  }

  /**
   * POST /api/wordfoundry/v1/nodes/{id}/restore - Restore a node.
   */
  public function restoreNode(int $id, Request $request): JsonResponse {
    if (!$this->authenticate($request)) {
      return $this->unauthorized();
    }

    $node = Node::load($id);
    if (!$node) {
      return new JsonResponse([
        'success' => FALSE,
        'error' => [
          'code' => 'NOT_FOUND',
          'message' => 'Node not found',
        ],
      ], 404);
    }

    if ($node->isPublished()) {
      return new JsonResponse([
        'success' => FALSE,
        'error' => [
          'code' => 'NOT_TRASHED',
          'message' => 'Node is not unpublished',
        ],
      ], 400);
    }

    $node->setPublished();

    // Mark as synced from WordFoundry (to avoid sync loop).
    if ($node->hasField('field_wordfoundry_synced')) {
      $node->set('field_wordfoundry_synced', time());
    }

    try {
      $node->save();
    }
    catch (\Exception $e) {
      return new JsonResponse([
        'success' => FALSE,
        'error' => [
          'code' => 'RESTORE_FAILED',
          'message' => $e->getMessage(),
        ],
      ], 500);
    }

    return new JsonResponse([
      'success' => TRUE,
      'post_id' => (int) $node->id(),
      'status' => 'publish',
      'permalink' => $node->toUrl()->setAbsolute()->toString(),
    ]);
  }

  /**
   * GET /api/wordfoundry/v1/nodes - List nodes for import.
   */
  public function listNodes(Request $request): JsonResponse {
    if (!$this->authenticate($request)) {
      return $this->unauthorized();
    }

    $settings = $this->settings->getAll();
    $contentType = $settings['content_type'] ?? 'article';

    $status = $request->query->get('status', 'any');
    $perPage = min((int) $request->query->get('per_page', 20), 100);
    $page = max(1, (int) $request->query->get('page', 1));
    $search = $request->query->get('search', '');
    $excludeImported = $request->query->get('exclude_imported', 'true') === 'true';

    // Build query.
    $query = $this->entityTypeManager
      ->getStorage('node')
      ->getQuery()
      ->accessCheck(FALSE)
      ->condition('type', $contentType)
      ->sort('changed', 'DESC')
      ->range(($page - 1) * $perPage, $perPage);

    if ($status !== 'any') {
      $query->condition('status', $status === 'publish' ? 1 : 0);
    }

    if (!empty($search)) {
      $query->condition('title', '%' . $search . '%', 'LIKE');
    }

    // Exclude already imported nodes.
    if ($excludeImported) {
      $query->notExists('field_wordfoundry_id');
    }

    $nodeIds = $query->execute();
    $nodes = Node::loadMultiple($nodeIds);

    // Count total.
    $countQuery = $this->entityTypeManager
      ->getStorage('node')
      ->getQuery()
      ->accessCheck(FALSE)
      ->condition('type', $contentType)
      ->count();

    if ($status !== 'any') {
      $countQuery->condition('status', $status === 'publish' ? 1 : 0);
    }
    if ($excludeImported) {
      $countQuery->notExists('field_wordfoundry_id');
    }

    $total = (int) $countQuery->execute();

    $posts = [];
    foreach ($nodes as $node) {
      $author = $node->getOwner();

      // Get featured image.
      $featuredImageUrl = '';
      if ($node->hasField('field_image') && !$node->get('field_image')->isEmpty()) {
        $file = $node->get('field_image')->entity;
        if ($file) {
          $featuredImageUrl = \Drupal::service('file_url_generator')
            ->generateAbsoluteString($file->getFileUri());
        }
      }

      // Get excerpt.
      $body = $node->hasField('body') ? $node->get('body')->value : '';
      $excerpt = strip_tags($body);
      $excerpt = mb_substr($excerpt, 0, 200) . (mb_strlen($excerpt) > 200 ? '...' : '');

      $posts[] = [
        'id' => (int) $node->id(),
        'title' => $node->getTitle(),
        'excerpt' => $excerpt,
        'status' => $node->isPublished() ? 'publish' : 'draft',
        'date' => date('c', $node->getCreatedTime()),
        'modified' => date('c', $node->getChangedTime()),
        'author' => $author ? $author->getDisplayName() : 'Unknown',
        'author_id' => $author ? (int) $author->id() : 0,
        'featured_image_url' => $featuredImageUrl,
        'permalink' => $node->toUrl()->setAbsolute()->toString(),
        'word_count' => str_word_count(strip_tags($body)),
      ];
    }

    return new JsonResponse([
      'success' => TRUE,
      'posts' => $posts,
      'total' => $total,
      'pages' => (int) ceil($total / $perPage),
      'current_page' => $page,
    ]);
  }

  /**
   * GET /api/wordfoundry/v1/nodes/{id}/export - Export a node.
   */
  public function exportNode(int $id, Request $request): JsonResponse {
    if (!$this->authenticate($request)) {
      return $this->unauthorized();
    }

    $node = Node::load($id);
    if (!$node) {
      return new JsonResponse([
        'success' => FALSE,
        'error' => [
          'code' => 'NOT_FOUND',
          'message' => 'Node not found',
        ],
      ], 404);
    }

    $author = $node->getOwner();

    // Get featured image.
    $featuredImageUrl = '';
    if ($node->hasField('field_image') && !$node->get('field_image')->isEmpty()) {
      $file = $node->get('field_image')->entity;
      if ($file) {
        $featuredImageUrl = \Drupal::service('file_url_generator')
          ->generateAbsoluteString($file->getFileUri());
      }
    }

    // Get body content.
    $body = $node->hasField('body') ? $node->get('body')->value : '';

    // Get taxonomy terms.
    $settings = $this->settings->getAll();
    $taxonomy = $settings['taxonomy'] ?? 'tags';
    $categories = [];
    $fieldName = $this->getTaxonomyFieldName($node->bundle(), $taxonomy);
    if ($fieldName && $node->hasField($fieldName)) {
      foreach ($node->get($fieldName)->referencedEntities() as $term) {
        $categories[] = [
          'id' => (int) $term->id(),
          'name' => $term->getName(),
          'slug' => $term->id(),
        ];
      }
    }

    return new JsonResponse([
      'success' => TRUE,
      'post' => [
        'id' => (int) $node->id(),
        'title' => $node->getTitle(),
        'content' => $body,
        'excerpt' => '',
        'status' => $node->isPublished() ? 'publish' : 'draft',
        'date' => date('c', $node->getCreatedTime()),
        'modified' => date('c', $node->getChangedTime()),
        'author' => $author ? $author->getDisplayName() : 'Unknown',
        'author_id' => $author ? (int) $author->id() : 0,
        'featured_image_url' => $featuredImageUrl,
        'permalink' => $node->toUrl()->setAbsolute()->toString(),
        'word_count' => str_word_count(strip_tags($body)),
        'categories' => $categories,
        'tags' => [],
        'meta_description' => '',
      ],
    ]);
  }

  /**
   * GET /api/wordfoundry/v1/nodes/similar - Find similar nodes.
   */
  public function similarNodes(Request $request): JsonResponse {
    if (!$this->authenticate($request)) {
      return $this->unauthorized();
    }

    $keywords = $request->query->get('keywords', '');
    $categoriesParam = $request->query->get('categories', '');
    $excludeId = (int) $request->query->get('exclude_id', 0);
    $limit = min((int) $request->query->get('limit', 10), 20);

    // Parse categories.
    $categoryIds = [];
    if (!empty($categoriesParam)) {
      $categoryIds = array_filter(array_map('intval', explode(',', $categoriesParam)));
    }

    // Parse keywords.
    $keywordList = [];
    if (!empty($keywords)) {
      $keywordList = array_filter(array_map('trim', explode(',', $keywords)));
    }

    if (empty($categoryIds) && empty($keywordList)) {
      return new JsonResponse(['posts' => []]);
    }

    $settings = $this->settings->getAll();
    $contentType = $settings['content_type'] ?? 'article';
    $taxonomy = $settings['taxonomy'] ?? 'tags';

    $foundPosts = [];
    $foundIds = [];

    // Search by category.
    if (!empty($categoryIds)) {
      $fieldName = $this->getTaxonomyFieldName($contentType, $taxonomy);
      if ($fieldName) {
        $query = $this->entityTypeManager
          ->getStorage('node')
          ->getQuery()
          ->accessCheck(FALSE)
          ->condition('type', $contentType)
          ->condition('status', 1)
          ->condition($fieldName, $categoryIds, 'IN')
          ->sort('created', 'DESC')
          ->range(0, $limit * 2);

        if ($excludeId) {
          $query->condition('nid', $excludeId, '!=');
        }

        $nodeIds = $query->execute();
        $nodes = Node::loadMultiple($nodeIds);

        foreach ($nodes as $node) {
          $nid = (int) $node->id();
          if (!in_array($nid, $foundIds)) {
            $foundIds[] = $nid;
            $foundPosts[$nid] = [
              'node' => $node,
              'relevance' => 'category',
              'score' => 2,
            ];
          }
        }
      }
    }

    // Search by keywords.
    if (!empty($keywordList)) {
      foreach ($keywordList as $keyword) {
        if (strlen($keyword) < 3) {
          continue;
        }

        $query = $this->entityTypeManager
          ->getStorage('node')
          ->getQuery()
          ->accessCheck(FALSE)
          ->condition('type', $contentType)
          ->condition('status', 1)
          ->condition('title', '%' . $keyword . '%', 'LIKE')
          ->range(0, $limit);

        if ($excludeId) {
          $query->condition('nid', $excludeId, '!=');
        }

        $nodeIds = $query->execute();
        $nodes = Node::loadMultiple($nodeIds);

        foreach ($nodes as $node) {
          $nid = (int) $node->id();
          if (in_array($nid, $foundIds)) {
            $foundPosts[$nid]['relevance'] = 'both';
            $foundPosts[$nid]['score'] = 3;
          }
          else {
            $foundIds[] = $nid;
            $foundPosts[$nid] = [
              'node' => $node,
              'relevance' => 'keyword',
              'score' => 1,
            ];
          }
        }
      }
    }

    // Sort by score.
    uasort($foundPosts, fn($a, $b) => $b['score'] - $a['score']);

    // Limit results.
    $foundPosts = array_slice($foundPosts, 0, $limit, TRUE);

    // Format response.
    $posts = [];
    foreach ($foundPosts as $item) {
      $node = $item['node'];
      $body = $node->hasField('body') ? $node->get('body')->value : '';
      $excerpt = strip_tags($body);
      $excerpt = mb_substr($excerpt, 0, 100) . (mb_strlen($excerpt) > 100 ? '...' : '');

      $posts[] = [
        'id' => (int) $node->id(),
        'title' => $node->getTitle(),
        'url' => $node->toUrl()->setAbsolute()->toString(),
        'excerpt' => $excerpt,
        'relevance' => $item['relevance'],
        'categories' => [],
        'date' => date('Y-m-d H:i:s', $node->getCreatedTime()),
      ];
    }

    return new JsonResponse(['posts' => $posts]);
  }

  /**
   * Get taxonomy field name for a content type.
   */
  protected function getTaxonomyFieldName(string $contentType, string $vocabulary): ?string {
    // Common field naming patterns.
    $possibleFields = [
      'field_' . $vocabulary,
      'field_' . str_replace('_', '', $vocabulary),
      'field_tags',
      'field_categories',
      'field_category',
    ];

    $fieldDefinitions = $this->entityTypeManager
      ->getStorage('node')
      ->create(['type' => $contentType])
      ->getFieldDefinitions();

    foreach ($possibleFields as $fieldName) {
      if (isset($fieldDefinitions[$fieldName])) {
        return $fieldName;
      }
    }

    // Search for any taxonomy reference field.
    foreach ($fieldDefinitions as $fieldName => $definition) {
      if ($definition->getType() === 'entity_reference') {
        $settings = $definition->getSettings();
        if (($settings['target_type'] ?? '') === 'taxonomy_term') {
          return $fieldName;
        }
      }
    }

    return NULL;
  }

}
